<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

if(!isActiveUser()) {
    header("Location: activate.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$user = $conn->query("SELECT * FROM users WHERE id = $user_id")->fetch_assoc();

// Get withdrawal methods from admin settings
$withdraw_methods = [
    [
        'id' => 'bkash',
        'name' => 'bKash',
        'icon' => 'fas fa-mobile-alt',
        'color' => 'bg-pink-100 text-pink-600',
        'min_amount' => 100,
        'fee_percentage' => 0,
        'processing_time' => '30 minutes'
    ],
    [
        'id' => 'nagad',
        'name' => 'Nagad',
        'icon' => 'fas fa-wallet',
        'color' => 'bg-red-100 text-red-600',
        'min_amount' => 100,
        'fee_percentage' => 0,
        'processing_time' => '30 minutes'
    ],
    [
        'id' => 'rocket',
        'name' => 'Rocket',
        'icon' => 'fas fa-rocket',
        'color' => 'bg-purple-100 text-purple-600',
        'min_amount' => 100,
        'fee_percentage' => 0,
        'processing_time' => '30 minutes'
    ],
    [
        'id' => 'bank',
        'name' => 'Bank Account',
        'icon' => 'fas fa-university',
        'color' => 'bg-blue-100 text-blue-600',
        'min_amount' => 1000,
        'fee_percentage' => 0,
        'processing_time' => '24 hours'
    ]
];

// Handle withdrawal request
if($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['submit_withdraw'])) {
    $amount = (float)$_POST['amount'];
    $method = sanitize($_POST['method']);
    $account_number = sanitize($_POST['account_number']);
    $account_name = sanitize($_POST['account_name']);
    
    // Get selected method details
    $selected_method = null;
    foreach($withdraw_methods as $m) {
        if($m['id'] == $method) {
            $selected_method = $m;
            break;
        }
    }
    
    if(!$selected_method) {
        $error = "Invalid withdrawal method";
    } elseif($amount < $selected_method['min_amount']) {
        $error = "Minimum withdrawal amount for {$selected_method['name']} is ৳{$selected_method['min_amount']}";
    } elseif($amount > $user['balance']) {
        $error = "Insufficient balance. Your available balance is ৳" . number_format($user['balance'], 2);
    } elseif(empty($account_number)) {
        $error = "Account number is required";
    } elseif(empty($account_name)) {
        $error = "Account name is required";
    } else {
        // Calculate fee and net amount
        $fee = ($amount * $selected_method['fee_percentage']) / 100;
        $net_amount = $amount - $fee;
        
        // Start transaction
        $conn->begin_transaction();
        
        try {
            // Deduct balance
            $conn->query("UPDATE users SET balance = balance - $amount WHERE id = $user_id");
            
            // Create withdrawal transaction
            $description = "Withdrawal via {$selected_method['name']} - Account: $account_number - Name: $account_name";
            if($fee > 0) {
                $description .= " - Fee: ৳" . number_format($fee, 2);
            }
            
            $stmt = $conn->prepare("INSERT INTO transactions (user_id, type, amount, description, status) VALUES (?, 'withdrawal', ?, ?, 'pending')");
            $stmt->bind_param("ids", $user_id, $amount, $description);
            $stmt->execute();
            
            // Send notification to admin
            $admin_notification = "New withdrawal request from {$user['username']} for ৳$amount via {$selected_method['name']}";
            $conn->query("INSERT INTO admin_notifications (message, type) VALUES ('$admin_notification', 'withdrawal')");
            
            $conn->commit();
            
            $success = "Withdrawal request submitted successfully! It will be processed within {$selected_method['processing_time']}.";
        } catch (Exception $e) {
            $conn->rollback();
            $error = "Failed to submit withdrawal request. Please try again.";
        }
    }
}

// Get pending withdrawals
$pending_withdrawals = $conn->query("
    SELECT * FROM transactions 
    WHERE user_id = $user_id AND type = 'withdrawal' AND status = 'pending'
    ORDER BY created_at DESC
");

// Get recent withdrawals
$recent_withdrawals = $conn->query("
    SELECT * FROM transactions 
    WHERE user_id = $user_id AND type = 'withdrawal' 
    ORDER BY created_at DESC 
    LIMIT 5
");

// Get total withdrawals
$total_withdrawn = $conn->query("
    SELECT SUM(amount) as total FROM transactions 
    WHERE user_id = $user_id AND type = 'withdrawal' AND status = 'completed'
")->fetch_assoc()['total'] ?? 0;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Withdraw - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
    <style>
        .method-card {
            transition: all 0.3s ease;
        }
        .method-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 15px 30px rgba(0,0,0,0.1);
        }
        .method-card.selected {
            border-color: #3B82F6;
            border-width: 2px;
        }
        .amount-option {
            transition: all 0.2s ease;
        }
        .amount-option:hover {
            transform: scale(1.05);
        }
        .amount-option.selected {
            transform: scale(1.05);
            border-color: #3B82F6;
            background-color: #EFF6FF;
        }
        .withdraw-process {
            position: relative;
        }
        .withdraw-process:before {
            content: '';
            position: absolute;
            top: 24px;
            left: 20px;
            width: 2px;
            height: calc(100% - 48px);
            background-color: #e5e7eb;
            z-index: 1;
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Include Header -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Breadcrumb -->
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="index.php" class="inline-flex items-center text-sm text-gray-700 hover:text-blue-600">
                                <i class="fas fa-home mr-2"></i> Home
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <a href="dashboard.php" class="ml-1 text-sm text-gray-700 hover:text-blue-600 md:ml-2">Dashboard</a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400 text-xs"></i>
                                <span class="ml-1 text-sm text-gray-500 md:ml-2 font-medium">Withdraw</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>
            
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-minus-circle mr-2"></i> Withdraw Funds
                </h1>
                <p class="text-gray-600">Withdraw money from your wallet balance</p>
            </div>
            
            <!-- Messages -->
            <?php if(isset($error)): ?>
                <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-lg animate__animated animate__shakeX">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-3 text-xl"></i>
                        <span class="text-red-700 font-medium"><?php echo $error; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if(isset($success)): ?>
                <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg animate__animated animate__fadeIn">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-xl"></i>
                        <span class="text-green-700 font-medium"><?php echo $success; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Left Column - Withdraw Form -->
                <div class="lg:col-span-2 space-y-8">
                    <!-- Balance Card -->
                    <div class="bg-gradient-to-r from-green-500 to-emerald-600 rounded-2xl shadow-lg p-8 text-white animate__animated animate__fadeInLeft">
                        <div class="flex flex-col md:flex-row items-center justify-between">
                            <div>
                                <h2 class="text-2xl font-bold mb-2">Available Balance</h2>
                                <p class="text-green-100 opacity-90">Ready for withdrawal</p>
                            </div>
                            <div class="mt-4 md:mt-0 text-center md:text-right">
                                <div class="text-5xl font-bold mb-2">৳<?php echo number_format($user['balance'], 2); ?></div>
                                <div class="text-green-100">
                                    <i class="fas fa-history mr-1"></i>
                                    Total Withdrawn: ৳<?php echo number_format($total_withdrawn, 2); ?>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Withdrawal Methods -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-blue-500 to-indigo-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-wallet mr-2"></i> Select Withdrawal Method
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
                                <?php foreach($withdraw_methods as $method): ?>
                                <div class="method-card cursor-pointer border-2 border-gray-200 rounded-xl p-4 text-center hover:border-blue-300 transition duration-300"
                                     onclick="selectMethod('<?php echo $method['id']; ?>')"
                                     id="method-<?php echo $method['id']; ?>">
                                    <div class="w-12 h-12 <?php echo $method['color']; ?> rounded-full flex items-center justify-center mx-auto mb-3">
                                        <i class="<?php echo $method['icon']; ?> text-xl"></i>
                                    </div>
                                    <h4 class="font-bold text-gray-800"><?php echo $method['name']; ?></h4>
                                    <p class="text-gray-600 text-xs mt-1">
                                        Min: ৳<?php echo number_format($method['min_amount']); ?>
                                    </p>
                                    <p class="text-gray-600 text-xs">
                                        <i class="fas fa-clock mr-1"></i>
                                        <?php echo $method['processing_time']; ?>
                                    </p>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            
                            <!-- Method Details -->
                            <div id="methodDetails" class="hidden">
                                <div class="bg-gray-50 rounded-xl p-6 mb-6">
                                    <div id="methodContent"></div>
                                </div>
                                
                                <!-- Amount Selection -->
                                <div class="mb-6">
                                    <h4 class="text-lg font-bold text-gray-800 mb-4">
                                        <i class="fas fa-coins mr-2"></i> Select Amount
                                    </h4>
                                    
                                    <!-- Quick Amounts -->
                                    <div class="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-5 gap-3 mb-6">
                                        <?php 
                                        $amounts = [100, 200, 500, 1000, 2000, 3000, 5000, 10000, 20000, 50000];
                                        foreach($amounts as $amount): 
                                        ?>
                                        <button type="button" 
                                                onclick="selectAmount(<?php echo $amount; ?>)"
                                                class="amount-option border-2 border-gray-200 rounded-lg p-4 text-center hover:border-blue-300 transition duration-300"
                                                id="amount-<?php echo $amount; ?>">
                                            <span class="text-lg font-bold text-gray-800">৳<?php echo number_format($amount); ?></span>
                                        </button>
                                        <?php endforeach; ?>
                                    </div>
                                    
                                    <!-- Custom Amount -->
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            Or Enter Custom Amount
                                        </label>
                                        <div class="relative">
                                            <span class="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-500 font-bold">৳</span>
                                            <input type="number" 
                                                   id="customAmount"
                                                   name="amount"
                                                   min="100" 
                                                   max="50000"
                                                   step="10"
                                                   class="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                   placeholder="Enter withdrawal amount"
                                                   oninput="selectCustomAmount(this.value)">
                                        </div>
                                        <p class="text-gray-500 text-xs mt-2">
                                            Minimum: ৳<span id="minAmount">100</span>, Maximum: ৳<?php echo min(50000, $user['balance']); ?>
                                        </p>
                                    </div>
                                    
                                    <!-- Amount Summary -->
                                    <div id="amountSummary" class="hidden mt-4 p-4 bg-blue-50 rounded-lg">
                                        <div class="space-y-2">
                                            <div class="flex justify-between">
                                                <span class="text-gray-700">Withdrawal Amount:</span>
                                                <span class="font-bold" id="summaryAmount">৳0.00</span>
                                            </div>
                                            <div class="flex justify-between">
                                                <span class="text-gray-700">Processing Fee:</span>
                                                <span class="font-bold text-green-600" id="summaryFee">৳0.00 (0%)</span>
                                            </div>
                                            <div class="flex justify-between border-t pt-2">
                                                <span class="text-gray-700 font-bold">You Will Receive:</span>
                                                <span class="text-2xl font-bold text-green-600" id="summaryNet">৳0.00</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Withdrawal Form -->
                                <form method="POST" action="" id="withdrawForm">
                                    <input type="hidden" name="method" id="selectedMethod">
                                    <input type="hidden" name="amount" id="selectedAmount">
                                    
                                    <div class="space-y-4">
                                        <div>
                                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                                Account Number
                                            </label>
                                            <input type="text" 
                                                   name="account_number"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                   placeholder="01XXXXXXXXX"
                                                   required>
                                            <p class="text-gray-500 text-xs mt-2" id="accountHint">
                                                Enter your mobile banking number
                                            </p>
                                        </div>
                                        
                                        <div>
                                            <label class="block text-gray-700 text-sm font-bold mb-2">
                                                Account Holder Name
                                            </label>
                                            <input type="text" 
                                                   name="account_name"
                                                   class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                                   placeholder="As per account"
                                                   required>
                                            <p class="text-gray-500 text-xs mt-2">
                                                Name must match with account registration
                                            </p>
                                        </div>
                                        
                                        <!-- Security Verification -->
                                        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                                            <h5 class="font-bold text-yellow-800 mb-2">
                                                <i class="fas fa-shield-alt mr-2"></i> Security Verification
                                            </h5>
                                            <div class="flex items-start">
                                                <input type="checkbox" id="verify" required
                                                       class="mt-1 mr-3 h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded">
                                                <label for="verify" class="text-yellow-700 text-sm">
                                                    I verify that the account details provided are correct and belong to me.
                                                    I understand that withdrawals cannot be reversed once processed.
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <!-- Submit Button -->
                                        <div class="pt-4">
                                            <button type="submit" name="submit_withdraw"
                                                    class="w-full bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-4 px-6 rounded-lg text-lg hover:from-green-600 hover:to-emerald-700 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105">
                                                <i class="fas fa-paper-plane mr-2"></i> Submit Withdrawal Request
                                            </button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Withdrawal Process -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft" style="animation-delay: 0.2s">
                        <div class="bg-gradient-to-r from-purple-500 to-pink-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-cogs mr-2"></i> Withdrawal Process
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="withdraw-process space-y-8">
                                <div class="flex items-start relative z-10">
                                    <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-blue-600 font-bold text-xl">1</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 text-lg mb-2">Submit Request</h4>
                                        <p class="text-gray-600">Fill the withdrawal form with correct account details</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start relative z-10">
                                    <div class="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-green-600 font-bold text-xl">2</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 text-lg mb-2">Admin Verification</h4>
                                        <p class="text-gray-600">Our team verifies your request and account details</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start relative z-10">
                                    <div class="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-yellow-600 font-bold text-xl">3</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 text-lg mb-2">Processing</h4>
                                        <p class="text-gray-600">Payment is processed through selected method</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-start relative z-10">
                                    <div class="w-12 h-12 bg-purple-100 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                                        <span class="text-purple-600 font-bold text-xl">4</span>
                                    </div>
                                    <div>
                                        <h4 class="font-bold text-gray-800 text-lg mb-2">Completion</h4>
                                        <p class="text-gray-600">Money is sent to your account within specified time</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Right Column - Pending & Recent Withdrawals -->
                <div class="space-y-8">
                    <!-- Pending Withdrawals -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight">
                        <div class="bg-gradient-to-r from-yellow-500 to-orange-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-clock mr-2"></i> Pending Withdrawals
                            </h3>
                        </div>
                        <div class="p-6">
                            <?php if($pending_withdrawals->num_rows > 0): ?>
                                <div class="space-y-4">
                                    <?php while($withdrawal = $pending_withdrawals->fetch_assoc()): ?>
                                    <div class="p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                                        <div class="flex justify-between items-start mb-2">
                                            <span class="font-bold text-gray-800">
                                                ৳<?php echo number_format($withdrawal['amount'], 2); ?>
                                            </span>
                                            <span class="px-2 py-1 bg-yellow-100 text-yellow-800 text-xs font-bold rounded">
                                                Pending
                                            </span>
                                        </div>
                                        <p class="text-gray-600 text-sm mb-2">
                                            <?php echo substr($withdrawal['description'], 0, 50); ?>...
                                        </p>
                                        <p class="text-gray-500 text-xs">
                                            <i class="far fa-clock mr-1"></i>
                                            <?php echo date('M d, h:i A', strtotime($withdrawal['created_at'])); ?>
                                        </p>
                                    </div>
                                    <?php endwhile; ?>
                                </div>
                            <?php else: ?>
                                <div class="text-center py-8">
                                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                        <i class="fas fa-check text-gray-400 text-2xl"></i>
                                    </div>
                                    <p class="text-gray-600">No pending withdrawals</p>
                                </div>
                            <?php endif; ?>
                            
                            <div class="mt-6">
                                <a href="transactions.php?type=withdrawal" 
                                   class="block text-center text-blue-600 hover:text-blue-800 font-medium">
                                    <i class="fas fa-list mr-1"></i> View All Transactions
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Recent Withdrawals -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-blue-500 to-indigo-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-history mr-2"></i> Recent Withdrawals
                            </h3>
                        </div>
                        <div class="p-6">
                            <?php if($recent_withdrawals->num_rows > 0): ?>
                                <div class="space-y-4">
                                    <?php while($withdrawal = $recent_withdrawals->fetch_assoc()): 
                                        $status_color = $withdrawal['status'] == 'completed' ? 'green' : 
                                                       ($withdrawal['status'] == 'pending' ? 'yellow' : 'red');
                                    ?>
                                    <div class="p-4 bg-gray-50 rounded-lg hover:bg-gray-100 transition duration-300">
                                        <div class="flex justify-between items-start mb-2">
                                            <span class="font-bold text-gray-800">
                                                ৳<?php echo number_format($withdrawal['amount'], 2); ?>
                                            </span>
                                            <span class="px-2 py-1 bg-<?php echo $status_color; ?>-100 text-<?php echo $status_color; ?>-800 text-xs font-bold rounded">
                                                <?php echo ucfirst($withdrawal['status']); ?>
                                            </span>
                                        </div>
                                        <p class="text-gray-600 text-sm mb-2 truncate">
                                            <?php echo substr($withdrawal['description'], 0, 50); ?>...
                                        </p>
                                        <p class="text-gray-500 text-xs">
                                            <?php echo date('M d, h:i A', strtotime($withdrawal['created_at'])); ?>
                                        </p>
                                    </div>
                                    <?php endwhile; ?>
                                </div>
                            <?php else: ?>
                                <div class="text-center py-8">
                                    <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                        <i class="fas fa-history text-gray-400 text-2xl"></i>
                                    </div>
                                    <p class="text-gray-600">No withdrawal history</p>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Withdrawal Policy -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight" style="animation-delay: 0.2s">
                        <div class="bg-gradient-to-r from-gray-700 to-gray-900 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-gavel mr-2"></i> Withdrawal Policy
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="space-y-3">
                                <div class="flex items-start">
                                    <i class="fas fa-check text-green-500 mt-1 mr-3"></i>
                                    <p class="text-gray-600 text-sm">Minimum withdrawal: ৳100</p>
                                </div>
                                <div class="flex items-start">
                                    <i class="fas fa-check text-green-500 mt-1 mr-3"></i>
                                    <p class="text-gray-600 text-sm">Processing time: 30 minutes to 24 hours</p>
                                </div>
                                <div class="flex items-start">
                                    <i class="fas fa-check text-green-500 mt-1 mr-3"></i>
                                    <p class="text-gray-600 text-sm">No withdrawal fee for now</p>
                                </div>
                                <div class="flex items-start">
                                    <i class="fas fa-check text-green-500 mt-1 mr-3"></i>
                                    <p class="text-gray-600 text-sm">Daily withdrawal limit: ৳50,000</p>
                                </div>
                                <div class="flex items-start">
                                    <i class="fas fa-check text-green-500 mt-1 mr-3"></i>
                                    <p class="text-gray-600 text-sm">Account must be verified</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Support Card -->
                    <div class="bg-gradient-to-r from-red-500 to-pink-600 rounded-xl shadow-lg p-6 text-white animate__animated animate__fadeInRight" style="animation-delay: 0.3s">
                        <h3 class="text-xl font-bold mb-4">
                            <i class="fas fa-headset mr-2"></i> Need Help?
                        </h3>
                        <p class="mb-6 opacity-90">
                            Having trouble with withdrawal? Contact our support team.
                        </p>
                        <div class="space-y-3">
                            <a href="https://t.me/support" target="_blank" 
                               class="flex items-center justify-center bg-white text-blue-600 font-bold py-3 px-4 rounded-lg hover:bg-blue-50 transition duration-300">
                                <i class="fab fa-telegram mr-2 text-xl"></i> Telegram Support
                            </a>
                            <a href="https://wa.me/8801234567890" target="_blank" 
                               class="flex items-center justify-center bg-white text-green-600 font-bold py-3 px-4 rounded-lg hover:bg-green-50 transition duration-300">
                                <i class="fab fa-whatsapp mr-2 text-xl"></i> WhatsApp Support
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    // Method selection
    let selectedMethod = null;
    let selectedAmount = 0;
    
    function selectMethod(methodId) {
        // Find method details
        const methods = {
            'bkash': {
                name: 'bKash',
                min_amount: 100,
                fee_percentage: 0,
                processing_time: '30 minutes',
                account_hint: 'Enter your bKash number (01XXXXXXXXX)',
                icon: 'fas fa-mobile-alt',
                color: 'pink'
            },
            'nagad': {
                name: 'Nagad',
                min_amount: 100,
                fee_percentage: 0,
                processing_time: '30 minutes',
                account_hint: 'Enter your Nagad number (01XXXXXXXXX)',
                icon: 'fas fa-wallet',
                color: 'red'
            },
            'rocket': {
                name: 'Rocket',
                min_amount: 100,
                fee_percentage: 0,
                processing_time: '30 minutes',
                account_hint: 'Enter your Rocket number (01XXXXXXXXX)',
                icon: 'fas fa-rocket',
                color: 'purple'
            },
            'bank': {
                name: 'Bank Account',
                min_amount: 1000,
                fee_percentage: 0,
                processing_time: '24 hours',
                account_hint: 'Enter your bank account number',
                icon: 'fas fa-university',
                color: 'blue'
            }
        };
        
        selectedMethod = methods[methodId];
        
        // Update UI
        document.querySelectorAll('.method-card').forEach(card => {
            card.classList.remove('selected');
        });
        document.getElementById('method-' + methodId).classList.add('selected');
        
        // Show method details
        document.getElementById('methodDetails').classList.remove('hidden');
        document.getElementById('selectedMethod').value = methodId;
        
        // Update method content
        document.getElementById('methodContent').innerHTML = `
            <div class="flex items-start">
                <div class="w-12 h-12 bg-${selectedMethod.color}-100 rounded-full flex items-center justify-center mr-4">
                    <i class="${selectedMethod.icon} text-${selectedMethod.color}-600 text-xl"></i>
                </div>
                <div>
                    <h4 class="text-xl font-bold text-gray-800 mb-1">${selectedMethod.name}</h4>
                    <div class="space-y-1 text-sm text-gray-600">
                        <p><i class="fas fa-coins mr-2"></i> Minimum: ৳${selectedMethod.min_amount.toLocaleString()}</p>
                        <p><i class="fas fa-clock mr-2"></i> Processing: ${selectedMethod.processing_time}</p>
                        <p><i class="fas fa-percentage mr-2"></i> Fee: ${selectedMethod.fee_percentage}%</p>
                    </div>
                </div>
            </div>
        `;
        
        // Update account hint
        document.getElementById('accountHint').textContent = selectedMethod.account_hint;
        
        // Update minimum amount
        document.getElementById('minAmount').textContent = selectedMethod.min_amount;
        document.getElementById('customAmount').min = selectedMethod.min_amount;
        document.getElementById('customAmount').placeholder = `Enter amount (${selectedMethod.min_amount} - ${Math.min(50000, <?php echo $user['balance']; ?>))}`;
        
        // Clear previous amount selection
        selectedAmount = 0;
        document.getElementById('selectedAmount').value = '';
        document.getElementById('customAmount').value = '';
        document.querySelectorAll('.amount-option').forEach(option => {
            option.classList.remove('selected');
        });
        document.getElementById('amountSummary').classList.add('hidden');
        
        // Scroll to form
        document.getElementById('methodDetails').scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
    
    // Amount selection
    function selectAmount(amount) {
        if(!selectedMethod) {
            showNotification('Please select a withdrawal method first', 'error');
            return;
        }
        
        if(amount < selectedMethod.min_amount) {
            showNotification(`Minimum amount for ${selectedMethod.name} is ৳${selectedMethod.min_amount}`, 'error');
            return;
        }
        
        if(amount > <?php echo $user['balance']; ?>) {
            showNotification('Insufficient balance', 'error');
            return;
        }
        
        selectedAmount = amount;
        
        // Update UI
        document.querySelectorAll('.amount-option').forEach(option => {
            option.classList.remove('selected');
        });
        document.getElementById('amount-' + amount).classList.add('selected');
        
        // Update form and summary
        updateAmountSummary();
    }
    
    function selectCustomAmount(value) {
        const amount = parseInt(value) || 0;
        
        if(!selectedMethod) {
            showNotification('Please select a withdrawal method first', 'error');
            document.getElementById('customAmount').value = '';
            return;
        }
        
        selectedAmount = amount;
        
        // Remove selection from predefined amounts
        document.querySelectorAll('.amount-option').forEach(option => {
            option.classList.remove('selected');
        });
        
        // Update form and summary
        updateAmountSummary();
    }
    
    function updateAmountSummary() {
        if(!selectedAmount || selectedAmount < selectedMethod.min_amount) {
            document.getElementById('amountSummary').classList.add('hidden');
            return;
        }
        
        // Calculate fee and net amount
        const fee = (selectedAmount * selectedMethod.fee_percentage) / 100;
        const netAmount = selectedAmount - fee;
        
        // Update summary
        document.getElementById('summaryAmount').textContent = '৳' + selectedAmount.toLocaleString();
        document.getElementById('summaryFee').textContent = '৳' + fee.toLocaleString() + ' (' + selectedMethod.fee_percentage + '%)';
        document.getElementById('summaryNet').textContent = '৳' + netAmount.toLocaleString();
        
        // Update hidden input
        document.getElementById('selectedAmount').value = selectedAmount;
        
        // Show summary
        document.getElementById('amountSummary').classList.remove('hidden');
    }
    
    // Form validation
    document.getElementById('withdrawForm').addEventListener('submit', function(e) {
        if(!selectedMethod) {
            e.preventDefault();
            showNotification('Please select a withdrawal method', 'error');
            return;
        }
        
        if(!selectedAmount || selectedAmount < selectedMethod.min_amount) {
            e.preventDefault();
            showNotification(`Please enter a valid amount (minimum ৳${selectedMethod.min_amount})`, 'error');
            return;
        }
        
        if(selectedAmount > <?php echo $user['balance']; ?>) {
            e.preventDefault();
            showNotification('Insufficient balance', 'error');
            return;
        }
        
        const accountNumber = this.querySelector('[name="account_number"]').value;
        const accountName = this.querySelector('[name="account_name"]').value;
        
        if(!accountNumber.trim()) {
            e.preventDefault();
            showNotification('Please enter account number', 'error');
            return;
        }
        
        if(!accountName.trim()) {
            e.preventDefault();
            showNotification('Please enter account name', 'error');
            return;
        }
        
        if(!confirm(`Are you sure you want to withdraw ৳${selectedAmount} to ${accountNumber} via ${selectedMethod.name}?`)) {
            e.preventDefault();
        }
    });
    
    function showNotification(message, type) {
        const notification = document.createElement('div');
        notification.className = `fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
        notification.innerHTML = `
            <div class="flex items-center">
                <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
                <span>${message}</span>
            </div>
        `;
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.classList.remove('animate__fadeInRight');
            notification.classList.add('animate__fadeOutRight');
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }
    
    // Add animations
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.animate__animated');
        cards.forEach((card, index) => {
            setTimeout(() => {
                card.classList.add('animate__animated', 'animate__fadeIn');
            }, index * 100);
        });
    });
    </script>
</body>
</html>